/******************************************************************************
 *  bwm-ng                                                                    *
 *                                                                            *
 *  Copyright (C) 2004 Volker Gropp (vgropp@pefra.de)                         *
 *                                                                            *
 *  for more info read README.                                                *
 *                                                                            *
 *  This program is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  This program is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with this program; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <getopt.h>
#include <limits.h>
#include <signal.h>
#include <string.h> 

/* following only for check_if_up and ioctl */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>

#define MAJOR 0
#define MINOR 3
/* used for this nice spinning wheel */
#define IDLE_CHARS "-\\|/"

struct iface_stats {
    char    *if_name;
    unsigned long long rec;
    unsigned long long send;
	unsigned long long p_rec;
	unsigned long long p_send;
};

/* global vars and options */
typedef struct iface_stats t_iface_stats;
int if_count=0;
char PROC_NET_DEV[PATH_MAX];
unsigned int delay=500;
char show_kb=1;
char netstat_i=0;
char show_all_if=0;
char show_packets=0;

/* fd for check_if_up and ioctl */
int skfd = -1;

/* global buffer to store all data of interfaces in */
t_iface_stats *if_stats=NULL;
/* total struct */
t_iface_stats if_stats_total;

/* clear stuff */
int deinit(char *error_msg) {
	/* first close curses, so we dont leave mess behind */
	endwin();
	curs_set(1);
	/* close socket if we opened it for ioctl */
	if (skfd >= 0) { 
		close(skfd);
	}
	/* we should clean if_state, i think, */
	if (if_stats!=NULL) free(if_stats);
	/* output errormsg if given */
	if (error_msg!=NULL) printf(error_msg);
	/* we are done, say goodbye */
    exit(0);
}


/* sigint handler */
void sigint(int sig) {
	/* we got a sigint, call deinit and exit */
	deinit(NULL);
}


/* convert doube bytes/s value to some nice string */
char *convert_bytes(double bytes,char * buffer, int buf_size) {
	if (bytes<1024 && !show_kb)
		snprintf(buffer,buf_size,"%12.2f  B/s",bytes);
	else if (bytes<1048576 || show_kb)
		snprintf(buffer,buf_size,"%12.2f KB/s",bytes/1024);
	else snprintf(buffer,buf_size,"%12.2f MB/s",bytes/1048576);
	return buffer;
}


/* test whether the iface is up or not */
char check_if_up(char *ifname) {
	struct ifreq ifr;
	if (skfd < 0) {
		/* maybe check some /proc file first like net-tools do */
	    if ((skfd =  socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) {
    	    deinit("socket error");
    	}
	}
	strcpy(ifr.ifr_name, ifname);
	if (ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0) {
		return 0; /* return if as down if there was some error */
	} 
	if (ifr.ifr_flags & IFF_UP) return 1; /* check against IFF_UP and return */
		else return 0;
}

/* do the actual work, get and print stats */
void get_iface_stats () {
    FILE *f;
    char buffer[1024],name[1024],bytes_buf[20];
	char *ptr;
	unsigned long long bytesr,bytess,packets_in,packets_out,buf;
    int local_if_count,hidden_if=0;
	float multiplier=(float)1000/delay;
    t_iface_stats stats; /* local struct, used to calc total values */
	memset(&stats,0,(size_t)sizeof(t_iface_stats)); /* init it */
	/* dont open proc_net_dev if netstat_i is requested, else try to open and if it fails fallback */
    if (netstat_i || !(f=fopen(PROC_NET_DEV,"r"))) {
		if (show_all_if) {
			if (!(f=popen("netstat -ia","r"))) {
				deinit("no input stream found");
			}
		} else {
			if (!(f=popen("netstat -i","r"))) {
				deinit("no input stream found");
			}
		}
		netstat_i=1;
	}
	if ((fgets(buffer,1023,f) != NULL ) && (fgets(buffer,1023,f) != NULL )) {
	    while (fgets(buffer,1023,f) != NULL ) {
			bytesr=bytess=packets_in=packets_out=0;
			if (!netstat_i) {
				/* get the name */
				ptr=strchr(buffer,':');
				/* wrong format */
				if (ptr==NULL) { deinit("wrong format of input stream"); }
				/* set : to end_of_string and move to first char of "next" string (to first data) */
				*ptr++ = 0;
				sscanf(ptr,"%llu%llu%llu%llu%llu%llu%llu%llu%llu%llu",&bytesr,&packets_in,&buf,&buf,&buf,&buf,&buf,&buf,&bytess,&packets_out);
				sscanf(buffer,"%s",name);
			} else {
				sscanf(buffer,"%s%llu%llu%llu%llu%llu%llu%llu%llu",name,&buf,&buf,&packets_in,&buf,&buf,&buf,&packets_out,&buf);
				bytesr=bytess=0;
			}
			for (local_if_count=0;local_if_count<if_count;local_if_count++) {
				/*check if its the correct if */
				if (!strcmp(name,if_stats[local_if_count].if_name)) break; 
			}
			if (local_if_count==if_count) { 
				/* iface not found, seems like there is a new one! */
				if_count++;
				if_stats=(t_iface_stats*)realloc(if_stats,sizeof(t_iface_stats)*if_count);
				if (name[0]!='\0') if_stats[if_count-1].if_name=(char*)strdup(name);
					else if_stats[if_count-1].if_name=(char*)strdup("unknown");
				/* set it to current value, so there is no peak at first announce */
	            if_stats[local_if_count].send=bytess;
		        if_stats[local_if_count].rec=bytesr;
				if_stats[local_if_count].p_send=packets_out;
				if_stats[local_if_count].p_rec=packets_in;
				if_stats_total.send+=bytess;
				if_stats_total.rec+=bytesr;
				if_stats_total.p_send+=packets_out;
				if_stats_total.p_rec+=packets_in;
			}
			if (show_all_if || check_if_up(name)) { /* is it up or do we show all ifaces? */
				mvwprintw(stdscr,5+local_if_count-hidden_if,8,"%12s:",name); /* output the name */
				if (netstat_i || show_packets) {
					/* show packets/s if asked for or netstat input */
					wprintw(stdscr,"%13.2f P/s ",(double)(packets_out-if_stats[local_if_count].p_send)*multiplier);
					wprintw(stdscr,"%13.2f P/s ",(double)(packets_in-if_stats[local_if_count].p_rec)*multiplier);
					wprintw(stdscr,"%13.2f P/s ",(double)(packets_out+packets_in-(if_stats[local_if_count].p_send+if_stats[local_if_count].p_rec))*multiplier);
				} else {
					/* output Bytes/s */
		            wprintw(stdscr,"%s ",convert_bytes((double)((bytess-if_stats[local_if_count].send)*multiplier),bytes_buf,20));
		            wprintw(stdscr,"%s ",convert_bytes((double)((bytesr-if_stats[local_if_count].rec)*multiplier),bytes_buf,20));
		            /* print total (send+rec) of current iface */
		            wprintw(stdscr,"%s",convert_bytes((double)((bytess+bytesr-(if_stats[local_if_count].send+if_stats[local_if_count].rec))*multiplier),bytes_buf,20));
				}
			} else hidden_if++; /* increase the opt cause we dont show this if */
            /* save current stats for the next run */
            if_stats[local_if_count].send=bytess;
            if_stats[local_if_count].rec=bytesr;
			if_stats[local_if_count].p_send=packets_out;
			if_stats[local_if_count].p_rec=packets_in;
            /* add current iface stats to total */
            stats.send+=bytess;
            stats.rec+=bytesr;
			stats.p_send+=packets_out;
			stats.p_rec+=packets_in;
		}
	}
    /* output total ifaces stats */
	mvwprintw(stdscr,6+local_if_count-hidden_if,8,"------------------------------------------------------------------");
    mvwprintw(stdscr,7+local_if_count-hidden_if,8,"%12s:","total");
	if (netstat_i || show_packets) {
		/* show packets/s if asked for or netstat input */
		wprintw(stdscr,"%13.2f P/s ",(double)(stats.p_send-if_stats_total.p_send)*multiplier);
		wprintw(stdscr,"%13.2f P/s ",(double)(stats.p_rec-if_stats_total.p_rec)*multiplier);
		/* output total of all ifaces */
		wprintw(stdscr,"%13.2f P/s ",(double)((stats.p_send-if_stats_total.p_send)+(stats.p_rec-if_stats_total.p_rec))*multiplier);
	} else {
		/* output Bytes/s */
	    wprintw(stdscr,"%s ",convert_bytes((double)((stats.send-if_stats_total.send)*multiplier),bytes_buf,20));
	    wprintw(stdscr,"%s ",convert_bytes((double)((stats.rec-if_stats_total.rec)*multiplier),bytes_buf,20));
	    /* output total of all ifaces */
	    wprintw(stdscr,"%s",convert_bytes((double)((stats.send+stats.rec-(if_stats_total.send+if_stats_total.rec))*multiplier),bytes_buf,20));
	}
	/* save the data in total-struct */
	if_stats_total.send=stats.send;
	if_stats_total.rec=stats.rec;
	if_stats_total.p_send=stats.p_send;
	if_stats_total.p_rec=stats.p_rec;
	/* close input stream */
	if (netstat_i) 
		pclose(f);
	else 
		fclose(f);
    return; 
}

/* prints a helpscreen and exists */
int printhelp() {
	printf("bwm-tng v%i.%i Copyright (C) 2004 Volker Gropp\n",MAJOR,MINOR);
	printf("USAGE: bwm-ng [OPTION] ...\n");
	printf("displays current ethernet interfaces load in kb/s\n\n");
	printf("Options:\n");
	printf("  -t, --timeout <msec>  displays stats every <msec> (1msec = 1/1000sec) default: 500\n");
	printf("  -f, --file <filename> filename to read raw data from. default: /proc/net/dev\n");
	printf("  -d, --dynamic         show values dynamicly (Byte KB or MB)\n");
	printf("  -a, --allif           show also down interfaces\n");
	printf("  -p, --packets         show packets/s instead of KB/s\n");
	printf("  -n, --netstat         always use netstat -i as input stream\n");
	printf("  -h, --help            displays this help\n");
	printf("  -V, --version         print version info\n");
	printf("\n");
	exit(0);
}

int main (int argc, char *argv[]) {
	char c='\0'; /* used for getch */ 
	char o;
	unsigned char idle_chars_p=0;
	strcpy(PROC_NET_DEV,"/proc/net/dev");
	/* get command line arguments, kinda ugly, wanna rewrite it? */
	while (1) {
		int option_index = 0;
		static struct option long_options[] = {
			{"timeout", 1, 0, 0},
			{"file",1,0,0},
			{"dynamic",0,0,0},
			{"netstat",0,0,0},
			{"help", 0, 0, 0},
			{"version",0,0,0},
			{"allif",0,0,0},
			{"packets",0,0,0},
			{0,0,0,0}
		};
		o=getopt_long (argc,argv,"ht:f:dnVap",long_options, &option_index);
		if (o==-1) break;
		switch (o) {
			case '?': exit(0);
					  break;
			case 0: if (!strcasecmp(long_options[option_index].name,"help")) printhelp();
						else if (!strcasecmp(long_options[option_index].name,"timeout")) {
							if ((optarg) && atol(optarg)>0) { delay=atol(optarg); }
						} else if (!strcasecmp(long_options[option_index].name,"file")) {
							if (optarg && (strlen(optarg)<PATH_MAX)) strcpy(PROC_NET_DEV,optarg);
						} else if (!strcasecmp(long_options[option_index].name,"dynamic")) {
							show_kb=0;
						} else if (!strcasecmp(long_options[option_index].name,"allif")) {
							show_all_if=1;
						} else if (!strcasecmp(long_options[option_index].name,"packets")) {
							show_packets=1;
						} else if (!strcasecmp(long_options[option_index].name,"netstat")) {
							netstat_i=1;
						} else if (!strcasecmp(long_options[option_index].name,"version")) {
							printf("Bandwidth Monitor NG (bmw-ng) v%i.%i Copyright (C) 2004 Volker Gropp\n",MAJOR,MINOR);
							exit(0);
						}
					break;
			case 'h':
				printhelp();
				break;
			case 'f':
				if (optarg && (strlen(optarg)<PATH_MAX)) strcpy(PROC_NET_DEV,optarg);
				break;
			case 'a':
				show_all_if=1;
				break;
			case 'p':
				show_packets=1;
				break;
			case 't':
				if ((optarg) && atol(optarg)>0) { delay=atol(optarg); } 
				break;
			case 'd':
				show_kb=0;
				break;
			case 'n':
				netstat_i=1;
				break;
			case 'V':
				printf("Bandwidth Monitor NG (bmw-ng) v%i.%i Copyright (C) 2004 Volker Gropp\n",MAJOR,MINOR);
				exit(0);
				break;
		}
	}
	memset(&if_stats_total,0,(size_t)sizeof(t_iface_stats));
	/* init curses */
	initscr(); 
	cbreak(); 
	noecho();
	nonl();
	curs_set(0);
	/* end of init curses, now set a sigint handler to deinit the screen on ctrl-break */
	signal(SIGINT,sigint);
	timeout(delay); /* set the timeout of getch to delay in ms) */
    while (1) { /* do the main loop */
		erase();
        mvwprintw(stdscr,1,8,"bwm-ng v%i.%i (delay %2.3fs), press 'q' to end this",MAJOR,MINOR,(float)delay/1000);
		if (netstat_i) 
			mvwprintw(stdscr,2,8,"netstat -i %s",show_all_if==1 ? "(all)" : "");
		else 
			mvwprintw(stdscr,2,8,"/proc/net/dev %s",show_all_if==1 ? "(all)" : "");
		mvwprintw(stdscr,3,8,"%c       iface               Tx                Rx             Total",(char)IDLE_CHARS[idle_chars_p]);
		/* go to next char for next run */
		idle_chars_p++;
		if (idle_chars_p>3) idle_chars_p=0;
		mvwprintw(stdscr,4,8,"==================================================================");
		get_iface_stats(); /* do the actual work, get and print stats */
		refresh();
		c=getch();
		switch (c) {
			/* lets check for known keys */
			case '+': /* increase delay */
				delay+=100;
				timeout(delay);
				break;
			case '-': /* decrease delay */
				if (delay>100) {
					delay+=-100;
					timeout(delay);
				}
				break;
			case 'p':
			case 'P':
				show_packets=!show_packets;
				break;
			case 'a':
			case 'A':
				show_all_if=!show_all_if;
				break;
			case 'n':
			case 'N':
				netstat_i=!netstat_i;
				/* switched input, reset iface stats */
				free(if_stats);
				if_stats=NULL;
				if_count=0;
				memset(&if_stats_total,0,(size_t)sizeof(t_iface_stats));
				break;
			case 'q':
			case 'Q':
				/* we are asked to exit */
				deinit(NULL);
				break;
			case 'd':
			case 'D':
			case 'k':
			case 'K':
				/* switch kilobyte/autoassign */
				show_kb=!show_kb;
				break;
		}
		c='\0'; /* not really needed, but i like it this way */
    }
	/* we prolly never get here */
	deinit(NULL);
}
