/******************************************************************************
 *  bwm-ng v0.1                                                               *
 *                                                                            *
 *  Copyright (C) 2004 Volker Gropp (vgropp@pefra.de)                         *
 *                                                                            *
 *  for more info read README.                                                *
 *                                                                            *
 *  This program is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  This program is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with this program; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <getopt.h>
#include <limits.h>
#include <signal.h>
#include <string.h>

#define MAJOR 0
#define MINOR 2

struct iface_stats {
    char    *if_name;
    unsigned long long rec;
    unsigned long long send;
};

typedef struct iface_stats t_iface_stats;
int if_count=0;
char PROC_NET_DEV[PATH_MAX];
unsigned int delay=500;
char show_kb=0;

/* global buffer to store all data of interfaces in */
t_iface_stats *if_stats=NULL;
/* total struct */
t_iface_stats if_stats_total;

/* clear stuff */
int deinit() {
	/* first close curses, so we dont leave mess behind */
	endwin();
	curs_set(1);
	/* we should clean if_state, i think, */
	if (if_stats!=NULL) free(if_stats);
	/* we are done, say goodbye */
    exit(0);
}


/* sigint handler */
void sigint(int sig) {
	/* we got a sigint, call deinit and exit */
	deinit();
}


char *convert_bytes(double bytes,char * buffer, int buf_size) {
	if (bytes<1024 && !show_kb)
		snprintf(buffer,buf_size,"%9.0f  Byte/s",bytes);
	else if (bytes<1048576 || show_kb)
		snprintf(buffer,buf_size,"%12.2f KB/s",bytes/1024);
	else snprintf(buffer,buf_size,"%12.2f MB/s",bytes/1048576);
	return buffer;
}


/* do the actual work, get and print stats */
void get_iface_stats () {
    FILE *f;
    char buffer[1024],name[1024],bytes_buf[20];
	char *ptr;
	unsigned long long bytesr,bytess,buf;
//	char buf[4096]; /* dummy buffer, anyone knows a better solution? */
    int local_if_count;
	float multiplier=(float)1000/delay;
    t_iface_stats stats; /* local struct, used to calc total values */
    stats.rec=0; /* init it */
    stats.send=0;
    if ((f=fopen(PROC_NET_DEV,"r"))) {
		if ((fgets(buffer,1023,f) != NULL ) && (fgets(buffer,1023,f) != NULL )) {
	        while (fgets(buffer,1023,f) != NULL ) {
				/* get the name */
				ptr=strchr(buffer,':');
				/* wrong format */
				if (ptr==NULL) { printf("wrong format of %s\n",PROC_NET_DEV); deinit(); }
				/* set : to end_of_string and move to first char of "next" string (to first data) */
				*ptr++ = 0;
				sscanf(ptr,"%llu%llu%llu%llu%llu%llu%llu%llu%llu",&bytesr,&buf,&buf,&buf,&buf,&buf,&buf,&buf,&bytess);
				sscanf(buffer,"%s",name);
				for (local_if_count=0;local_if_count<if_count;local_if_count++) {
					/*check if its the correct if */
					if (!strcmp(name,if_stats[local_if_count].if_name)) break; 
				}
				if (local_if_count==if_count) { 
					/* iface not found, seems like there is a new one! */
					if_count++;
					if_stats=(t_iface_stats*)realloc(if_stats,sizeof(t_iface_stats)*if_count);
					if (name[0]!='\0') if_stats[if_count-1].if_name=(char*)strdup(name);
						else if_stats[if_count-1].if_name=(char*)strdup("unknown");
					/* set it to current value, so there is no peak at first announce */
	                if_stats[local_if_count].send=bytess;
		            if_stats[local_if_count].rec=bytesr;
					if_stats_total.send+=bytess;
					if_stats_total.rec+=bytesr;
				}
                mvwprintw(stdscr,5+local_if_count,8,"%12s:",name); /* output the name */
                /* output it */
                wprintw(stdscr,"%s ",convert_bytes((double)((bytess-if_stats[local_if_count].send)*multiplier),bytes_buf,20));
                wprintw(stdscr,"%s ",convert_bytes((double)((bytesr-if_stats[local_if_count].rec)*multiplier),bytes_buf,20));
                /* print total (send+rec) of current iface */
                wprintw(stdscr,"%s\n",convert_bytes((double)((bytess+bytesr-(if_stats[local_if_count].send+if_stats[local_if_count].rec))*multiplier),bytes_buf,20));
                /* save current stats for the next run */
                if_stats[local_if_count].send=bytess;
                if_stats[local_if_count].rec=bytesr;
                /* add current iface stats to total */
                stats.send+=bytess;
                stats.rec+=bytesr;
			}
		}
        /* output total ifaces stats */
		mvwprintw(stdscr,6+local_if_count,8,"------------------------------------------------------------------");
        mvwprintw(stdscr,7+local_if_count,8,"%12s:","total");
        wprintw(stdscr,"%s ",convert_bytes((double)((stats.send-if_stats_total.send)*multiplier),bytes_buf,20));
        wprintw(stdscr,"%s ",convert_bytes((double)((stats.rec-if_stats_total.rec)*multiplier),bytes_buf,20));
        /* output total of all ifaces */
        wprintw(stdscr,"%s\n",convert_bytes((double)((stats.send+stats.rec-(if_stats_total.send+if_stats_total.rec))*multiplier),bytes_buf,20));
		/* save the data in total-struct */
		if_stats_total.send=stats.send;
		if_stats_total.rec=stats.rec;
        fclose(f);
    }
    return; 
}

/* prints a helpscreen and exists */
int printhelp() {
	printf("bwm-tng v%i.%i Copyright (C) 2004 Volker Gropp\n",MAJOR,MINOR);
	printf("USAGE: bwm-ng [OPTION] ...\n");
	printf("displays current ethernet interfaces load in kb/s\n\n");
	printf("Options:\n");
	printf("  -t, --timeout <msec>  displays stats every <msec> (1msec = 1/1000sec) default: 500\n");
	printf("  -f, --file <filename> filename to read raw data from. default: /proc/net/dev\n");
	printf("  -k                    always show values as KB\n");
	printf("  -h, --help            displays this help\n");
	printf("\n");
	exit(0);
}

int main (int argc, char *argv[]) {
	char c='\0'; /* used for getch */ 
	char o;
	strcpy(PROC_NET_DEV,"/proc/net/dev");
	/* get command line arguments, kinda ugly, wanna rewrite it? */
	while (1) {
		int option_index = 0;
		static struct option long_options[] = {
			{"timeout", 1, 0, 0},
			{"file",1,0,0},
			{"help", 0, 0, 0},
			{0,0,0,0}
		};
		o=getopt_long (argc,argv,"ht:f:k",long_options, &option_index);
		if (o==-1) break;
		switch (o) {
			case '?': exit(0);
					  break;
			case 0: if (!strcasecmp(long_options[option_index].name,"help")) printhelp();
						else if (!strcasecmp(long_options[option_index].name,"timeout")) {
							if ((optarg) && atol(optarg)>0) { delay=atol(optarg); }
						} else if (!strcasecmp(long_options[option_index].name,"file")) {
							if (optarg && (strlen(optarg)<PATH_MAX)) strcpy(PROC_NET_DEV,optarg);
						}
					break;
			case 'h':
				printhelp();
				break;
			case 'f':
				if (optarg && (strlen(optarg)<PATH_MAX)) strcpy(PROC_NET_DEV,optarg);
				break;
			case 't':
				if ((optarg) && atol(optarg)>0) { delay=atol(optarg); } 
				break;
			case 'k':
				show_kb=1;
				break;
		}
	}
	/* init curses */
	initscr(); 
	cbreak(); 
	noecho();
	nonl();
	curs_set(0);
	/* end of init curses, now set a sigint handler to deinit the screen on ctrl-break */
	signal(SIGINT,sigint);
	timeout(delay); /* set the timeout of getch to delay in ms) */
    while (1) { /* do the main loop */
		erase();
        mvwprintw(stdscr,1,8,"bwm-ng v%i.%i (delay %2.3fs), press 'q' to end this",MAJOR,MINOR,(float)delay/1000);
		mvwprintw(stdscr,3,16,"iface               Tx                Rx             Total");
		mvwprintw(stdscr,4,8,"==================================================================");
		get_iface_stats(); /* do the actual work, get and print stats */
		refresh();
		c=getch();
		switch (c) {
			/* lets check for known keys */
			case '+': /* increase delay */
				delay+=100;
				timeout(delay);
				break;
			case '-': /* decrease delay */
				if (delay>100) {
					delay+=-100;
					timeout(delay);
				}
				break;
			case 'q':
			case 'Q':
				/* we are asked to exit */
				deinit();
				break;
			case 'k':
			case 'K':
				/* switch kilobyte/autoassign */
				show_kb=!show_kb;
				break;
		}
		c='\0'; /* not really needed, but i like it this way */
    }
	/* we prolly never get here */
	deinit();
}
